<?php
require_once 'config.php';

$action = $_GET['action'] ?? '';

try {
    $pdo = getDbConnection();
    $userId = getUserId(); // You'll need to implement user authentication
    
    switch ($action) {
        case 'addresses':
            getAddresses($pdo, $userId);
            break;
            
        case 'add':
            addAddress($pdo, $userId);
            break;
            
        case 'update':
            updateAddress($pdo, $userId);
            break;
            
        case 'delete':
            deleteAddress($pdo, $userId);
            break;
            
        case 'set-default':
            setDefaultAddress($pdo, $userId);
            break;
            
        default:
            handleError('Invalid action', 400);
    }
} catch (Exception $e) {
    handleError('Server error: ' . $e->getMessage(), 500);
}

function getUserId() {
    // Simple user ID for demo - in production, implement proper JWT authentication
    return 1;
}

function getAddresses($pdo, $userId) {
    $stmt = $pdo->prepare("SELECT * FROM addresses WHERE user_id = :user_id ORDER BY is_default DESC, created_at DESC");
    $stmt->execute(['user_id' => $userId]);
    $addresses = $stmt->fetchAll();
    
    // Format data
    foreach ($addresses as &$address) {
        $address['is_default'] = (bool)$address['is_default'];
        $address['created_at'] = date('Y-m-d H:i:s', strtotime($address['created_at']));
        $address['updated_at'] = date('Y-m-d H:i:s', strtotime($address['updated_at']));
    }
    
    sendResponse(true, 'Addresses retrieved successfully', $addresses);
}

function addAddress($pdo, $userId) {
    $data = getRequestData();
    validateRequired($data, ['name', 'phone', 'address_line_1', 'city', 'state', 'pincode']);
    
    $name = $data['name'];
    $phone = $data['phone'];
    $addressLine1 = $data['address_line_1'];
    $addressLine2 = $data['address_line_2'] ?? '';
    $city = $data['city'];
    $state = $data['state'];
    $pincode = $data['pincode'];
    $landmark = $data['landmark'] ?? '';
    $addressType = $data['address_type'] ?? 'home';
    $isDefault = $data['is_default'] ?? false;
    
    // If this is set as default, unset other default addresses
    if ($isDefault) {
        $stmt = $pdo->prepare("UPDATE addresses SET is_default = 0 WHERE user_id = :user_id");
        $stmt->execute(['user_id' => $userId]);
    }
    
    $stmt = $pdo->prepare("
        INSERT INTO addresses (
            user_id, name, phone, address_line_1, address_line_2, city, state, 
            pincode, landmark, address_type, is_default, created_at, updated_at
        ) VALUES (
            :user_id, :name, :phone, :address_line_1, :address_line_2, :city, :state,
            :pincode, :landmark, :address_type, :is_default, NOW(), NOW()
        )
    ");
    
    $stmt->execute([
        'user_id' => $userId,
        'name' => $name,
        'phone' => $phone,
        'address_line_1' => $addressLine1,
        'address_line_2' => $addressLine2,
        'city' => $city,
        'state' => $state,
        'pincode' => $pincode,
        'landmark' => $landmark,
        'address_type' => $addressType,
        'is_default' => $isDefault ? 1 : 0
    ]);
    
    $addressId = $pdo->lastInsertId();
    
    // Get the created address
    $stmt = $pdo->prepare("SELECT * FROM addresses WHERE id = :id");
    $stmt->execute(['id' => $addressId]);
    $address = $stmt->fetch();
    
    $address['is_default'] = (bool)$address['is_default'];
    
    sendResponse(true, 'Address added successfully', $address);
}

function updateAddress($pdo, $userId) {
    $data = getRequestData();
    validateRequired($data, ['id', 'name', 'phone', 'address_line_1', 'city', 'state', 'pincode']);
    
    $addressId = $data['id'];
    $name = $data['name'];
    $phone = $data['phone'];
    $addressLine1 = $data['address_line_1'];
    $addressLine2 = $data['address_line_2'] ?? '';
    $city = $data['city'];
    $state = $data['state'];
    $pincode = $data['pincode'];
    $landmark = $data['landmark'] ?? '';
    $addressType = $data['address_type'] ?? 'home';
    $isDefault = $data['is_default'] ?? false;
    
    // Check if address exists and belongs to user
    $stmt = $pdo->prepare("SELECT * FROM addresses WHERE id = :id AND user_id = :user_id");
    $stmt->execute(['id' => $addressId, 'user_id' => $userId]);
    $existingAddress = $stmt->fetch();
    
    if (!$existingAddress) {
        handleError('Address not found', 404);
    }
    
    // If this is set as default, unset other default addresses
    if ($isDefault) {
        $stmt = $pdo->prepare("UPDATE addresses SET is_default = 0 WHERE user_id = :user_id AND id != :id");
        $stmt->execute(['user_id' => $userId, 'id' => $addressId]);
    }
    
    $stmt = $pdo->prepare("
        UPDATE addresses SET 
            name = :name,
            phone = :phone,
            address_line_1 = :address_line_1,
            address_line_2 = :address_line_2,
            city = :city,
            state = :state,
            pincode = :pincode,
            landmark = :landmark,
            address_type = :address_type,
            is_default = :is_default,
            updated_at = NOW()
        WHERE id = :id AND user_id = :user_id
    ");
    
    $stmt->execute([
        'id' => $addressId,
        'name' => $name,
        'phone' => $phone,
        'address_line_1' => $addressLine1,
        'address_line_2' => $addressLine2,
        'city' => $city,
        'state' => $state,
        'pincode' => $pincode,
        'landmark' => $landmark,
        'address_type' => $addressType,
        'is_default' => $isDefault ? 1 : 0
    ]);
    
    // Get the updated address
    $stmt = $pdo->prepare("SELECT * FROM addresses WHERE id = :id");
    $stmt->execute(['id' => $addressId]);
    $address = $stmt->fetch();
    
    $address['is_default'] = (bool)$address['is_default'];
    
    sendResponse(true, 'Address updated successfully', $address);
}

function deleteAddress($pdo, $userId) {
    $addressId = $_GET['id'] ?? '';
    
    if (empty($addressId)) {
        handleError('Address ID is required', 400);
    }
    
    // Check if address exists and belongs to user
    $stmt = $pdo->prepare("SELECT * FROM addresses WHERE id = :id AND user_id = :user_id");
    $stmt->execute(['id' => $addressId, 'user_id' => $userId]);
    $address = $stmt->fetch();
    
    if (!$address) {
        handleError('Address not found', 404);
    }
    
    // Delete the address
    $stmt = $pdo->prepare("DELETE FROM addresses WHERE id = :id AND user_id = :user_id");
    $stmt->execute(['id' => $addressId, 'user_id' => $userId]);
    
    sendResponse(true, 'Address deleted successfully');
}

function setDefaultAddress($pdo, $userId) {
    $addressId = $_GET['id'] ?? '';
    
    if (empty($addressId)) {
        handleError('Address ID is required', 400);
    }
    
    // Check if address exists and belongs to user
    $stmt = $pdo->prepare("SELECT * FROM addresses WHERE id = :id AND user_id = :user_id");
    $stmt->execute(['id' => $addressId, 'user_id' => $userId]);
    $address = $stmt->fetch();
    
    if (!$address) {
        handleError('Address not found', 404);
    }
    
    // Unset all default addresses for this user
    $stmt = $pdo->prepare("UPDATE addresses SET is_default = 0 WHERE user_id = :user_id");
    $stmt->execute(['user_id' => $userId]);
    
    // Set this address as default
    $stmt = $pdo->prepare("UPDATE addresses SET is_default = 1 WHERE id = :id");
    $stmt->execute(['id' => $addressId]);
    
    sendResponse(true, 'Default address updated successfully');
}
?>
