<?php
require_once 'config.php';

$action = $_GET['action'] ?? '';

try {
    $pdo = getDbConnection();
    $userId = getUserId(); // You'll need to implement user authentication
    
    switch ($action) {
        case 'cart':
            getCart($pdo, $userId);
            break;
            
        case 'add':
            addToCart($pdo, $userId);
            break;
            
        case 'update':
            updateCartItem($pdo, $userId);
            break;
            
        case 'remove':
            removeFromCart($pdo, $userId);
            break;
            
        case 'clear':
            clearCart($pdo, $userId);
            break;
            
        default:
            handleError('Invalid action', 400);
    }
} catch (Exception $e) {
    handleError('Server error: ' . $e->getMessage(), 500);
}

function getUserId() {
    // Simple user ID for demo - in production, implement proper JWT authentication
    return 1;
}

function getCart($pdo, $userId) {
    $stmt = $pdo->prepare("
        SELECT c.*, p.name, p.price, p.discount_price, p.image_url, p.stock
        FROM cart c
        LEFT JOIN products p ON c.product_id = p.id
        WHERE c.user_id = :user_id AND p.is_active = 1
        ORDER BY c.created_at DESC
    ");
    $stmt->execute(['user_id' => $userId]);
    $cartItems = $stmt->fetchAll();
    
    // Calculate totals and format data
    $totalAmount = 0;
    $totalItems = 0;
    
    foreach ($cartItems as &$item) {
        $price = floatval($item['discount_price'] > 0 ? $item['discount_price'] : $item['price']);
        $item['total_price'] = $price * intval($item['quantity']);
        $item['price'] = floatval($item['price']);
        $item['discount_price'] = floatval($item['discount_price']);
        $item['quantity'] = intval($item['quantity']);
        $item['stock'] = intval($item['stock']);
        
        if (!empty($item['image_url'])) {
            $item['image_url'] = 'https://kwickbills.com/shop/api_core/images/products/' . $item['image_url'];
        }
        
        $totalAmount += $item['total_price'];
        $totalItems += $item['quantity'];
    }
    
    $response = [
        'items' => $cartItems,
        'summary' => [
            'total_items' => $totalItems,
            'total_amount' => $totalAmount
        ]
    ];
    
    sendResponse(true, 'Cart retrieved successfully', $response);
}

function addToCart($pdo, $userId) {
    $data = getRequestData();
    validateRequired($data, ['product_id', 'quantity']);
    
    $productId = $data['product_id'];
    $quantity = intval($data['quantity']);
    
    // Check if product exists and is active
    $stmt = $pdo->prepare("SELECT * FROM products WHERE id = :id AND is_active = 1");
    $stmt->execute(['id' => $productId]);
    $product = $stmt->fetch();
    
    if (!$product) {
        handleError('Product not found', 404);
    }
    
    // Check stock availability
    if ($product['stock'] < $quantity) {
        handleError('Insufficient stock available', 400);
    }
    
    // Check if item already exists in cart
    $stmt = $pdo->prepare("SELECT * FROM cart WHERE user_id = :user_id AND product_id = :product_id");
    $stmt->execute(['user_id' => $userId, 'product_id' => $productId]);
    $existingItem = $stmt->fetch();
    
    if ($existingItem) {
        // Update quantity
        $newQuantity = $existingItem['quantity'] + $quantity;
        
        if ($product['stock'] < $newQuantity) {
            handleError('Insufficient stock available', 400);
        }
        
        $stmt = $pdo->prepare("UPDATE cart SET quantity = :quantity, updated_at = NOW() WHERE id = :id");
        $stmt->execute(['quantity' => $newQuantity, 'id' => $existingItem['id']]);
        
        sendResponse(true, 'Cart updated successfully', ['cart_item_id' => $existingItem['id']]);
    } else {
        // Add new item
        $stmt = $pdo->prepare("INSERT INTO cart (user_id, product_id, quantity, created_at, updated_at) VALUES (:user_id, :product_id, :quantity, NOW(), NOW())");
        $stmt->execute([
            'user_id' => $userId,
            'product_id' => $productId,
            'quantity' => $quantity
        ]);
        
        sendResponse(true, 'Item added to cart successfully', ['cart_item_id' => $pdo->lastInsertId()]);
    }
}

function updateCartItem($pdo, $userId) {
    $data = getRequestData();
    validateRequired($data, ['cart_item_id', 'quantity']);
    
    $cartItemId = $data['cart_item_id'];
    $quantity = intval($data['quantity']);
    
    if ($quantity <= 0) {
        handleError('Quantity must be greater than 0', 400);
    }
    
    // Get cart item with product info
    $stmt = $pdo->prepare("
        SELECT c.*, p.stock 
        FROM cart c
        LEFT JOIN products p ON c.product_id = p.id
        WHERE c.id = :id AND c.user_id = :user_id
    ");
    $stmt->execute(['id' => $cartItemId, 'user_id' => $userId]);
    $cartItem = $stmt->fetch();
    
    if (!$cartItem) {
        handleError('Cart item not found', 404);
    }
    
    if ($cartItem['stock'] < $quantity) {
        handleError('Insufficient stock available', 400);
    }
    
    $stmt = $pdo->prepare("UPDATE cart SET quantity = :quantity, updated_at = NOW() WHERE id = :id");
    $stmt->execute(['quantity' => $quantity, 'id' => $cartItemId]);
    
    sendResponse(true, 'Cart item updated successfully');
}

function removeFromCart($pdo, $userId) {
    $cartItemId = $_GET['id'] ?? '';
    
    if (empty($cartItemId)) {
        handleError('Cart item ID is required', 400);
    }
    
    $stmt = $pdo->prepare("DELETE FROM cart WHERE id = :id AND user_id = :user_id");
    $result = $stmt->execute(['id' => $cartItemId, 'user_id' => $userId]);
    
    if ($stmt->rowCount() === 0) {
        handleError('Cart item not found', 404);
    }
    
    sendResponse(true, 'Item removed from cart successfully');
}

function clearCart($pdo, $userId) {
    $stmt = $pdo->prepare("DELETE FROM cart WHERE user_id = :user_id");
    $stmt->execute(['user_id' => $userId]);
    
    sendResponse(true, 'Cart cleared successfully');
}
?>
