<?php
require_once 'config.php';

$action = $_GET['action'] ?? '';

try {
    $pdo = getDbConnection();
    $userId = getUserId(); // You'll need to implement user authentication
    
    switch ($action) {
        case 'place':
            placeOrder($pdo, $userId);
            break;
            
        case 'orders':
            getOrders($pdo, $userId);
            break;
            
        case 'order':
            getOrder($pdo, $userId);
            break;
            
        case 'cancel':
            cancelOrder($pdo, $userId);
            break;
            
        case 'track':
            trackOrder($pdo, $userId);
            break;
            
        default:
            handleError('Invalid action', 400);
    }
} catch (Exception $e) {
    handleError('Server error: ' . $e->getMessage(), 500);
}

function getUserId() {
    // Simple user ID for demo - in production, implement proper JWT authentication
    return 1;
}

function placeOrder($pdo, $userId) {
    $data = getRequestData();
    validateRequired($data, ['address_id', 'payment_method', 'items']);
    
    $addressId = $data['address_id'];
    $paymentMethod = $data['payment_method'];
    $items = $data['items'];
    $deliverySlot = $data['delivery_slot'] ?? null;
    $notes = $data['notes'] ?? '';
    
    // Validate address
    $stmt = $pdo->prepare("SELECT * FROM addresses WHERE id = :id AND user_id = :user_id");
    $stmt->execute(['id' => $addressId, 'user_id' => $userId]);
    $address = $stmt->fetch();
    
    if (!$address) {
        handleError('Address not found', 404);
    }
    
    // Validate cart items and calculate total
    $totalAmount = 0;
    $orderItems = [];
    
    foreach ($items as $item) {
        $stmt = $pdo->prepare("SELECT * FROM products WHERE id = :id AND is_active = 1");
        $stmt->execute(['id' => $item['product_id']]);
        $product = $stmt->fetch();
        
        if (!$product) {
            handleError('Product not found: ' . $item['product_id'], 404);
        }
        
        if ($product['stock'] < $item['quantity']) {
            handleError('Insufficient stock for product: ' . $product['name'], 400);
        }
        
        $price = floatval($product['discount_price'] > 0 ? $product['discount_price'] : $product['price']);
        $itemTotal = $price * intval($item['quantity']);
        $totalAmount += $itemTotal;
        
        $orderItems[] = [
            'product' => $product,
            'quantity' => intval($item['quantity']),
            'price' => $price,
            'total' => $itemTotal
        ];
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // Create order
        $orderNumber = 'ORD' . date('YmdHis') . rand(1000, 9999);
        $stmt = $pdo->prepare("
            INSERT INTO orders (
                order_number, user_id, address_id, total_amount, payment_method, 
                delivery_slot, notes, status, created_at, updated_at
            ) VALUES (
                :order_number, :user_id, :address_id, :total_amount, :payment_method,
                :delivery_slot, :notes, 'placed', NOW(), NOW()
            )
        ");
        $stmt->execute([
            'order_number' => $orderNumber,
            'user_id' => $userId,
            'address_id' => $addressId,
            'total_amount' => $totalAmount,
            'payment_method' => $paymentMethod,
            'delivery_slot' => $deliverySlot,
            'notes' => $notes
        ]);
        
        $orderId = $pdo->lastInsertId();
        
        // Create order items
        $stmt = $pdo->prepare("
            INSERT INTO order_items (
                order_id, product_id, quantity, price, total_price, created_at
            ) VALUES (
                :order_id, :product_id, :quantity, :price, :total_price, NOW()
            )
        ");
        
        foreach ($orderItems as $item) {
            $stmt->execute([
                'order_id' => $orderId,
                'product_id' => $item['product']['id'],
                'quantity' => $item['quantity'],
                'price' => $item['price'],
                'total_price' => $item['total']
            ]);
            
            // Update product stock
            $updateStmt = $pdo->prepare("UPDATE products SET stock = stock - :quantity WHERE id = :id");
            $updateStmt->execute(['quantity' => $item['quantity'], 'id' => $item['product']['id']]);
        }
        
        // Clear cart
        $stmt = $pdo->prepare("DELETE FROM cart WHERE user_id = :user_id");
        $stmt->execute(['user_id' => $userId]);
        
        $pdo->commit();
        
        sendResponse(true, 'Order placed successfully', [
            'order_id' => $orderId,
            'order_number' => $orderNumber,
            'total_amount' => $totalAmount
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function getOrders($pdo, $userId) {
    $status = $_GET['status'] ?? null;
    
    $whereClause = "WHERE o.user_id = :user_id";
    $params = ['user_id' => $userId];
    
    if ($status) {
        $whereClause .= " AND o.status = :status";
        $params['status'] = $status;
    }
    
    $stmt = $pdo->prepare("
        SELECT o.*, a.name as address_name, a.address_line_1, a.address_line_2, a.city, a.state, a.pincode
        FROM orders o
        LEFT JOIN addresses a ON o.address_id = a.id
        $whereClause
        ORDER BY o.created_at DESC
    ");
    $stmt->execute($params);
    $orders = $stmt->fetchAll();
    
    // Format order data
    foreach ($orders as &$order) {
        $order['total_amount'] = floatval($order['total_amount']);
        $order['created_at'] = date('Y-m-d H:i:s', strtotime($order['created_at']));
        $order['updated_at'] = date('Y-m-d H:i:s', strtotime($order['updated_at']));
    }
    
    sendResponse(true, 'Orders retrieved successfully', $orders);
}

function getOrder($pdo, $userId) {
    $orderId = $_GET['id'] ?? '';
    
    if (empty($orderId)) {
        handleError('Order ID is required', 400);
    }
    
    // Get order details
    $stmt = $pdo->prepare("
        SELECT o.*, a.name as address_name, a.address_line_1, a.address_line_2, a.city, a.state, a.pincode, a.phone
        FROM orders o
        LEFT JOIN addresses a ON o.address_id = a.id
        WHERE o.id = :id AND o.user_id = :user_id
    ");
    $stmt->execute(['id' => $orderId, 'user_id' => $userId]);
    $order = $stmt->fetch();
    
    if (!$order) {
        handleError('Order not found', 404);
    }
    
    // Get order items
    $stmt = $pdo->prepare("
        SELECT oi.*, p.name as product_name, p.image_url
        FROM order_items oi
        LEFT JOIN products p ON oi.product_id = p.id
        WHERE oi.order_id = :order_id
    ");
    $stmt->execute(['order_id' => $orderId]);
    $orderItems = $stmt->fetchAll();
    
    // Format data
    $order['total_amount'] = floatval($order['total_amount']);
    $order['created_at'] = date('Y-m-d H:i:s', strtotime($order['created_at']));
    $order['updated_at'] = date('Y-m-d H:i:s', strtotime($order['updated_at']));
    
    foreach ($orderItems as &$item) {
        $item['price'] = floatval($item['price']);
        $item['total_price'] = floatval($item['total_price']);
        $item['quantity'] = intval($item['quantity']);
        
        if (!empty($item['image_url'])) {
            $item['image_url'] = 'https://kwickbills.com/shop/api_core/images/products/' . $item['image_url'];
        }
    }
    
    $order['items'] = $orderItems;
    
    sendResponse(true, 'Order retrieved successfully', $order);
}

function cancelOrder($pdo, $userId) {
    $data = getRequestData();
    validateRequired($data, ['order_id', 'reason']);
    
    $orderId = $data['order_id'];
    $reason = $data['reason'];
    
    // Check if order exists and can be cancelled
    $stmt = $pdo->prepare("SELECT * FROM orders WHERE id = :id AND user_id = :user_id");
    $stmt->execute(['id' => $orderId, 'user_id' => $userId]);
    $order = $stmt->fetch();
    
    if (!$order) {
        handleError('Order not found', 404);
    }
    
    if (!in_array($order['status'], ['placed', 'confirmed'])) {
        handleError('Order cannot be cancelled in current status', 400);
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // Update order status
        $stmt = $pdo->prepare("UPDATE orders SET status = 'cancelled', cancelled_at = NOW(), cancel_reason = :reason WHERE id = :id");
        $stmt->execute(['reason' => $reason, 'id' => $orderId]);
        
        // Restore product stock
        $stmt = $pdo->prepare("
            SELECT oi.product_id, oi.quantity 
            FROM order_items oi 
            WHERE oi.order_id = :order_id
        ");
        $stmt->execute(['order_id' => $orderId]);
        $orderItems = $stmt->fetchAll();
        
        foreach ($orderItems as $item) {
            $updateStmt = $pdo->prepare("UPDATE products SET stock = stock + :quantity WHERE id = :id");
            $updateStmt->execute(['quantity' => $item['quantity'], 'id' => $item['product_id']]);
        }
        
        $pdo->commit();
        
        sendResponse(true, 'Order cancelled successfully');
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function trackOrder($pdo, $userId) {
    $orderId = $_GET['id'] ?? '';
    
    if (empty($orderId)) {
        handleError('Order ID is required', 400);
    }
    
    // Get order with tracking info
    $stmt = $pdo->prepare("
        SELECT o.*, a.address_line_1, a.city, a.state, a.pincode
        FROM orders o
        LEFT JOIN addresses a ON o.address_id = a.id
        WHERE o.id = :id AND o.user_id = :user_id
    ");
    $stmt->execute(['id' => $orderId, 'user_id' => $userId]);
    $order = $stmt->fetch();
    
    if (!$order) {
        handleError('Order not found', 404);
    }
    
    // Mock tracking data - in production, integrate with real tracking service
    $trackingData = [
        'order_id' => $order['id'],
        'order_number' => $order['order_number'],
        'status' => $order['status'],
        'current_location' => [
            'lat' => 19.0760 + (rand(-10, 10) / 1000), // Mock coordinates around Mumbai
            'lng' => 72.8777 + (rand(-10, 10) / 1000)
        ],
        'estimated_delivery' => $order['delivery_slot'] ?? date('Y-m-d H:i:s', strtotime('+1 hour')),
        'tracking_history' => [
            [
                'status' => 'placed',
                'timestamp' => $order['created_at'],
                'message' => 'Order placed successfully'
            ]
        ]
    ];
    
    // Add status-specific tracking info
    if (in_array($order['status'], ['confirmed', 'preparing', 'out_for_delivery', 'delivered'])) {
        $trackingData['tracking_history'][] = [
            'status' => 'confirmed',
            'timestamp' => date('Y-m-d H:i:s', strtotime($order['created_at'] . ' +30 minutes')),
            'message' => 'Order confirmed and being prepared'
        ];
    }
    
    if (in_array($order['status'], ['out_for_delivery', 'delivered'])) {
        $trackingData['tracking_history'][] = [
            'status' => 'out_for_delivery',
            'timestamp' => date('Y-m-d H:i:s', strtotime($order['created_at'] . ' +1 hour')),
            'message' => 'Order out for delivery'
        ];
    }
    
    if ($order['status'] === 'delivered') {
        $trackingData['tracking_history'][] = [
            'status' => 'delivered',
            'timestamp' => $order['updated_at'],
            'message' => 'Order delivered successfully'
        ];
    }
    
    sendResponse(true, 'Tracking information retrieved successfully', $trackingData);
}
?>
