<?php
require_once 'config.php';

$action = $_GET['action'] ?? '';

try {
    $pdo = getDbConnection();
    
    switch ($action) {
        case 'banners':
            getBanners($pdo);
            break;
            
        case 'categories':
            getCategories($pdo);
            break;
            
        case 'products':
            getProducts($pdo);
            break;
            
        case 'product':
            getProduct($pdo);
            break;
            
        case 'search':
            searchProducts($pdo);
            break;
            
        default:
            handleError('Invalid action', 400);
    }
} catch (Exception $e) {
    handleError('Server error: ' . $e->getMessage(), 500);
}

function getBanners($pdo) {
    $stmt = $pdo->prepare("SELECT * FROM banners WHERE is_active = 1 ORDER BY display_order ASC");
    $stmt->execute();
    $banners = $stmt->fetchAll();
    
    // Add full image URLs
    foreach ($banners as &$banner) {
        $banner['image_url'] = 'https://kwickbills.com/shop/api_core/images/banners/' . $banner['image_url'];
    }
    
    sendResponse(true, 'Banners retrieved successfully', $banners);
}

function getCategories($pdo) {
    $stmt = $pdo->prepare("SELECT * FROM categories WHERE is_active = 1 ORDER BY display_order ASC");
    $stmt->execute();
    $categories = $stmt->fetchAll();
    
    // Add full image URLs
    foreach ($categories as &$category) {
        if (!empty($category['image_url'])) {
            $category['image_url'] = 'https://kwickbills.com/shop/api_core/images/categories/' . $category['image_url'];
        }
    }
    
    sendResponse(true, 'Categories retrieved successfully', $categories);
}

function getProducts($pdo) {
    $page = intval($_GET['page'] ?? 1);
    $limit = intval($_GET['limit'] ?? 20);
    $categoryId = $_GET['category_id'] ?? null;
    $offset = ($page - 1) * $limit;
    
    $whereClause = "WHERE p.is_active = 1";
    $params = [];
    
    if ($categoryId) {
        $whereClause .= " AND p.category_id = :category_id";
        $params['category_id'] = $categoryId;
    }
    
    // Get total count
    $countSql = "SELECT COUNT(*) as total FROM products p $whereClause";
    $countStmt = $pdo->prepare($countSql);
    $countStmt->execute($params);
    $total = $countStmt->fetch()['total'];
    
    // Get products
    $sql = "SELECT p.*, c.name as category_name 
            FROM products p 
            LEFT JOIN categories c ON p.category_id = c.id 
            $whereClause 
            ORDER BY p.created_at DESC 
            LIMIT :limit OFFSET :offset";
    
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    
    foreach ($params as $key => $value) {
        $stmt->bindValue(":$key", $value);
    }
    
    $stmt->execute();
    $products = $stmt->fetchAll();
    
    // Add full image URLs and format data
    foreach ($products as &$product) {
        if (!empty($product['image_url'])) {
            $product['image_url'] = 'https://kwickbills.com/shop/api_core/images/products/' . $product['image_url'];
        }
        $product['price'] = floatval($product['price']);
        $product['discount_price'] = floatval($product['discount_price']);
        $product['stock'] = intval($product['stock']);
    }
    
    $response = [
        'products' => $products,
        'pagination' => [
            'current_page' => $page,
            'per_page' => $limit,
            'total' => intval($total),
            'total_pages' => ceil($total / $limit)
        ]
    ];
    
    sendResponse(true, 'Products retrieved successfully', $response);
}

function getProduct($pdo) {
    $productId = $_GET['id'] ?? '';
    
    if (empty($productId)) {
        handleError('Product ID is required', 400);
    }
    
    $stmt = $pdo->prepare("
        SELECT p.*, c.name as category_name 
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        WHERE p.id = :id AND p.is_active = 1
    ");
    $stmt->execute(['id' => $productId]);
    $product = $stmt->fetch();
    
    if (!$product) {
        handleError('Product not found', 404);
    }
    
    // Add full image URLs
    if (!empty($product['image_url'])) {
        $product['image_url'] = 'https://kwickbills.com/shop/api_core/images/products/' . $product['image_url'];
    }
    
    $product['price'] = floatval($product['price']);
    $product['discount_price'] = floatval($product['discount_price']);
    $product['stock'] = intval($product['stock']);
    
    sendResponse(true, 'Product retrieved successfully', $product);
}

function searchProducts($pdo) {
    $query = $_GET['q'] ?? '';
    
    if (empty($query)) {
        handleError('Search query is required', 400);
    }
    
    $searchTerm = "%$query%";
    $stmt = $pdo->prepare("
        SELECT p.*, c.name as category_name 
        FROM products p 
        LEFT JOIN categories c ON p.category_id = c.id 
        WHERE p.is_active = 1 
        AND (p.name LIKE :query OR p.description LIKE :query)
        ORDER BY p.name ASC
        LIMIT 50
    ");
    $stmt->execute(['query' => $searchTerm]);
    $products = $stmt->fetchAll();
    
    // Add full image URLs
    foreach ($products as &$product) {
        if (!empty($product['image_url'])) {
            $product['image_url'] = 'https://kwickbills.com/shop/api_core/images/products/' . $product['image_url'];
        }
        $product['price'] = floatval($product['price']);
        $product['discount_price'] = floatval($product['discount_price']);
        $product['stock'] = intval($product['stock']);
    }
    
    sendResponse(true, 'Search results retrieved successfully', $products);
}
?>
